<?php
/**
 * LeanPress Control Panel - Complete English Version
 * WordPress Optimized Fork with JavaScript Optimizations
 * 
 * @author Riccardo Bastillo - Lean Bunker
 * @version 2.0
 * @license GPL-2.0+
 */

if (!defined('ABSPATH')) {
    exit;
}

// ============================================================================
// 1. CONSTANTS
// ============================================================================

define('LEANPRESS_VERSION', '2.0');
define('LEANPRESS_OPTION', 'leanpress_settings');

// ============================================================================
// 2. DEFAULT SETTINGS
// ============================================================================

function leanpress_get_default_settings() {
    return array(
        // 🔌 Editor
        'disable_gutenberg' => true,
        'disable_tinymce' => true,
        'use_html_editor' => true,
        
        // 🗑️ Libraries
        'disable_jquery_ui' => true,
        'lazy_load_jquery' => true,
        'remove_jquery_migrate' => true,
        'remove_wp_embed' => true,
        
        // 💬 Comments
        'disable_comments' => true,
        
        // 🗑️ RSS Feeds
        'disable_rss' => true,
        'disable_simplepie' => true,
        
        // ❤️ Heartbeat
        'disable_heartbeat' => true,
        
        // 🚫 Repository
        'disable_wporg_repository' => true,
        
        // 🔒 External Security
        'block_external_connections' => true,
        'disable_xmlrpc' => true,
        'remove_wp_generator' => true,
        'disable_oembed' => true,
        'disable_wp_cron' => true,
        'disable_app_passwords' => true,
        'disable_recovery_mode' => true,
        'disable_site_health' => true,
        
        // 🔌 REST API
        'disable_rest_api' => true,
        
        // 🔄 Updates
        'disable_auto_updates' => true,
        'disable_update_checks' => true,
        'hide_update_notices' => true,
        
        // 🗑️ Additional Libraries
        'disable_ixr' => true,
        
        // 🚫 WordPress Branding
        'remove_wp_branding' => true,
        'remove_dashboard' => true,
        
        // 🔗 Upload Redirect
        'redirect_plugin_upload' => true,
        'redirect_theme_upload' => true,
        
        // 🗑️ Revisions
        'disable_revisions' => true,
        
        // 🚫 Admin Bar
        'disable_admin_bar' => true,
        
        // 📉 Dashboard Widgets
        'remove_dashboard_widgets' => true,
        
        // ❓ Help & Screen Options
        'disable_help_tabs' => true,
        'disable_screen_options' => true,
        
        // 🎯 Pointer JS
        'disable_pointer_js' => true,
        
        // 🎨 Color Picker
        'disable_color_picker' => true,
        
        // 🔐 Password Meter
        'disable_password_meter' => true,
        
        // 📄 Text Diff
        'disable_text_diff' => true,
        
        // 🔄 JSON2
        'disable_json2' => true,
        
        // 🧩 Useless Widgets
        'remove_useless_widgets' => true,
    );
}

// ============================================================================
// 3. SETTINGS MANAGEMENT
// ============================================================================

function leanpress_get_settings() {
    $defaults = leanpress_get_default_settings();
    $saved = get_option(LEANPRESS_OPTION, array());
    return wp_parse_args($saved, $defaults);
}

function leanpress_is_enabled($feature) {
    static $settings = null;
    if ($settings === null) {
        $settings = leanpress_get_settings();
    }
    return isset($settings[$feature]) && $settings[$feature];
}

// ============================================================================
// 4. ADMIN MENU
// ============================================================================

add_action('admin_menu', 'leanpress_admin_menu');

function leanpress_admin_menu() {
    add_menu_page(
        'LeanPress Control Panel',
        'LeanPress',
        'manage_options',
        'leanpress-control-panel',
        'leanpress_control_panel_page',
        'dashicons-shield',
        1
    );
}

// ============================================================================
// 5. CONTROL PANEL PAGE
// ============================================================================

function leanpress_control_panel_page() {
    $settings = leanpress_get_settings();
    
    // Handle save
    if (isset($_POST['leanpress_save']) && check_admin_referer('leanpress_save_settings')) {
        $new_settings = leanpress_get_default_settings();
        
        foreach ($new_settings as $key => $default) {
            $new_settings[$key] = isset($_POST[$key]) ? true : false;
        }
        
        update_option(LEANPRESS_OPTION, $new_settings);
        
        echo '<div class="notice notice-success is-dismissible"><p><strong>✅ Settings saved successfully!</strong></p></div>';
        
        $settings = $new_settings;
    }
    
    // Handle reset
    if (isset($_GET['leanpress_reset']) && check_admin_referer('leanpress_reset_settings')) {
        delete_option(LEANPRESS_OPTION);
        echo '<div class="notice notice-success is-dismissible"><p><strong>✅ Settings restored to default values!</strong></p></div>';
        $settings = leanpress_get_default_settings();
    }
    
    // Calculate statistics
    $total_features = count($settings);
    $enabled_count = array_sum($settings);
    $disabled_count = $total_features - $enabled_count;
    $optimization_score = round(($disabled_count / $total_features) * 100);
    
    ?>
    <div class="wrap">
        <h1>🛡️ <strong>LeanPress Control Panel</strong></h1>
        <p class="description">Optimized WordPress Fork - Complete control of all features</p>
        
        <div class="notice notice-info">
            <p><strong>ℹ️ Information:</strong> This panel controls all LeanPress features. Settings are saved in the database and survive WordPress updates. To restore default settings, use the "Restore Defaults" button at the bottom of the page.</p>
        </div>
        
        <div class="leanpress-stats">
            <div class="postbox">
                <div class="inside">
                    <div class="leanpress-stat-box" style="text-align: center; padding: 20px;">
                        <div style="font-size: 36px; font-weight: bold; margin-bottom: 10px;"><?php echo $disabled_count; ?>/<?php echo $total_features; ?></div>
                        <div style="font-size: 14px; color: #646970; text-transform: uppercase; letter-spacing: 1px;">Disabled Features</div>
                    </div>
                </div>
            </div>
            
            <div class="postbox">
                <div class="inside">
                    <div class="leanpress-stat-box" style="text-align: center; padding: 20px;">
                        <div style="font-size: 36px; font-weight: bold; margin-bottom: 10px;"><?php echo $optimization_score; ?>%</div>
                        <div style="font-size: 14px; color: #646970; text-transform: uppercase; letter-spacing: 1px;">Optimization</div>
                    </div>
                </div>
            </div>
            
            <div class="postbox">
                <div class="inside">
                    <div class="leanpress-stat-box" style="text-align: center; padding: 20px;">
                        <div style="font-size: 36px; font-weight: bold; margin-bottom: 10px;">
                            <?php 
                            $stars = floor($optimization_score / 20);
                            echo str_repeat('★', $stars) . str_repeat('☆', 5 - $stars);
                            ?>
                        </div>
                        <div style="font-size: 14px; color: #646970; text-transform: uppercase; letter-spacing: 1px;">Security</div>
                    </div>
                </div>
            </div>
        </div>
        
        <form method="post" action="">
            <?php wp_nonce_field('leanpress_save_settings'); ?>
            
            <!-- 🔌 EDITOR -->
            <div class="postbox">
                <button type="button" class="handlediv" aria-expanded="true">
                    <span class="screen-reader-text">Toggle panel</span>
                    <span class="toggle-indicator" aria-hidden="true"></span>
                </button>
                <h2 class="hndle"><span>🔌 Editor & Content</span></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="disable_gutenberg">Disable Gutenberg</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_gutenberg" id="disable_gutenberg" value="1" <?php checked($settings['disable_gutenberg']); ?>>
                                        <strong>Disable Block Editor (Gutenberg)</strong><br>
                                        <span class="description">Completely removes the block editor and all its dependencies</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="disable_tinymce">Disable TinyMCE</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_tinymce" id="disable_tinymce" value="1" <?php checked($settings['disable_tinymce']); ?>>
                                        <strong>Disable WYSIWYG Editor (TinyMCE)</strong><br>
                                        <span class="description">Removes the visual TinyMCE editor</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="use_html_editor">Pure HTML Editor</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="use_html_editor" id="use_html_editor" value="1" <?php checked($settings['use_html_editor']); ?>>
                                        <strong>Use pure HTML editor</strong><br>
                                        <span class="description">Replaces the editor with a simple HTML textarea (requires TinyMCE disabled)</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- 🗑️ LIBRARIES -->
            <div class="postbox">
                <button type="button" class="handlediv" aria-expanded="true">
                    <span class="screen-reader-text">Toggle panel</span>
                    <span class="toggle-indicator" aria-hidden="true"></span>
                </button>
                <h2 class="hndle"><span>🗑️ JavaScript Libraries</span></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="disable_jquery_ui">Disable jQuery UI</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_jquery_ui" id="disable_jquery_ui" value="1" <?php checked($settings['disable_jquery_ui']); ?>>
                                        <strong>Disable jQuery UI</strong><br>
                                        <span class="description">Removes all jQuery UI components (datepicker, dialog, tabs, accordion, etc.)</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="lazy_load_jquery">Lazy Load jQuery</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="lazy_load_jquery" id="lazy_load_jquery" value="1" <?php checked($settings['lazy_load_jquery']); ?>>
                                        <strong>Lazy load jQuery</strong><br>
                                        <span class="description">Loads jQuery only when needed by other scripts (~89 KB saved on simple pages)</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="remove_jquery_migrate">Remove jQuery Migrate</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="remove_jquery_migrate" id="remove_jquery_migrate" value="1" <?php checked($settings['remove_jquery_migrate']); ?>>
                                        <strong>Remove jQuery Migrate</strong><br>
                                        <span class="description">Removes the obsolete jQuery Migrate library (~8 KB)</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="remove_wp_embed">Remove wp-embed</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="remove_wp_embed" id="remove_wp_embed" value="1" <?php checked($settings['remove_wp_embed']); ?>>
                                        <strong>Remove wp-embed</strong><br>
                                        <span class="description">Removes the wp-embed script not needed in most cases (~5 KB)</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- 💬 COMMENTS -->
            <div class="postbox">
                <button type="button" class="handlediv" aria-expanded="true">
                    <span class="screen-reader-text">Toggle panel</span>
                    <span class="toggle-indicator" aria-hidden="true"></span>
                </button>
                <h2 class="hndle"><span>💬 Comments & Social</span></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="disable_comments">Disable Comments</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_comments" id="disable_comments" value="1" <?php checked($settings['disable_comments']); ?>>
                                        <strong>Disable comment system</strong><br>
                                        <span class="description">Completely blocks comments on all posts and removes the comments menu</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- 🗑️ RSS FEEDS -->
            <div class="postbox">
                <button type="button" class="handlediv" aria-expanded="true">
                    <span class="screen-reader-text">Toggle panel</span>
                    <span class="toggle-indicator" aria-hidden="true"></span>
                </button>
                <h2 class="hndle"><span>🗑️ RSS Feeds & Syndication</span></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="disable_rss">Disable RSS Feeds</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_rss" id="disable_rss" value="1" <?php checked($settings['disable_rss']); ?>>
                                        <strong>Disable RSS Feeds</strong><br>
                                        <span class="description">Blocks all RSS feeds (posts, comments, etc.) and removes links from headers</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="disable_simplepie">Disable SimplePie</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_simplepie" id="disable_simplepie" value="1" <?php checked($settings['disable_simplepie']); ?>>
                                        <strong>Disable SimplePie</strong><br>
                                        <span class="description">Removes the SimplePie library for feed parsing</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- ❤️ HEARTBEAT -->
            <div class="postbox">
                <button type="button" class="handlediv" aria-expanded="true">
                    <span class="screen-reader-text">Toggle panel</span>
                    <span class="toggle-indicator" aria-hidden="true"></span>
                </button>
                <h2 class="hndle"><span>❤️ Heartbeat API</span></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="disable_heartbeat">Disable Heartbeat</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_heartbeat" id="disable_heartbeat" value="1" <?php checked($settings['disable_heartbeat']); ?>>
                                        <strong>Disable Heartbeat API</strong><br>
                                        <span class="description">Blocks AJAX polling every 60 seconds that consumes server resources</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- 🚫 REPOSITORY -->
            <div class="postbox">
                <button type="button" class="handlediv" aria-expanded="true">
                    <span class="screen-reader-text">Toggle panel</span>
                    <span class="toggle-indicator" aria-hidden="true"></span>
                </button>
                <h2 class="hndle"><span>🚫 WordPress.org Repository</span></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="disable_wporg_repository">Disable Repository</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_wporg_repository" id="disable_wporg_repository" value="1" <?php checked($settings['disable_wporg_repository']); ?>>
                                        <strong>Disable WP.org repository</strong><br>
                                        <span class="description">Blocks access to the official plugin and theme repository, shows only manual upload</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- 🔒 EXTERNAL SECURITY -->
            <div class="postbox">
                <button type="button" class="handlediv" aria-expanded="true">
                    <span class="screen-reader-text">Toggle panel</span>
                    <span class="toggle-indicator" aria-hidden="true"></span>
                </button>
                <h2 class="hndle"><span>🔒 Security & External Connections</span></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="block_external_connections">Block External Connections</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="block_external_connections" id="block_external_connections" value="1" <?php checked($settings['block_external_connections']); ?>>
                                        <strong>Block external connections</strong><br>
                                        <span class="description">Blocks all connections to wordpress.org, gravatar.com, CDNs and other external domains</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="disable_xmlrpc">Disable XML-RPC</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_xmlrpc" id="disable_xmlrpc" value="1" <?php checked($settings['disable_xmlrpc']); ?>>
                                        <strong>Disable XML-RPC</strong><br>
                                        <span class="description">Completely blocks the XML-RPC endpoint (vulnerable to attacks)</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="remove_wp_generator">Remove WP Generator</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="remove_wp_generator" id="remove_wp_generator" value="1" <?php checked($settings['remove_wp_generator']); ?>>
                                        <strong>Remove WP Generator Meta</strong><br>
                                        <span class="description">Hides the WordPress version from meta tags for security</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="disable_oembed">Disable oEmbed</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_oembed" id="disable_oembed" value="1" <?php checked($settings['disable_oembed']); ?>>
                                        <strong>Disable oEmbed</strong><br>
                                        <span class="description">Blocks the embed system for videos, tweets, etc.</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="disable_wp_cron">Disable WP-Cron</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_wp_cron" id="disable_wp_cron" value="1" <?php checked($settings['disable_wp_cron']); ?>>
                                        <strong>Disable WP-Cron</strong><br>
                                        <span class="description">Disables WordPress internal cron system (use real server cron)</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="disable_app_passwords">Disable Application Passwords</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_app_passwords" id="disable_app_passwords" value="1" <?php checked($settings['disable_app_passwords']); ?>>
                                        <strong>Disable Application Passwords</strong><br>
                                        <span class="description">Blocks the application password system</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="disable_recovery_mode">Disable Recovery Mode</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_recovery_mode" id="disable_recovery_mode" value="1" <?php checked($settings['disable_recovery_mode']); ?>>
                                        <strong>Disable Recovery Mode</strong><br>
                                        <span class="description">Disables recovery mode for fatal errors</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="disable_site_health">Disable Site Health</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_site_health" id="disable_site_health" value="1" <?php checked($settings['disable_site_health']); ?>>
                                        <strong>Disable Site Health</strong><br>
                                        <span class="description">Removes the Site Health page from dashboard</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- 🔌 REST API -->
            <div class="postbox">
                <button type="button" class="handlediv" aria-expanded="true">
                    <span class="screen-reader-text">Toggle panel</span>
                    <span class="toggle-indicator" aria-hidden="true"></span>
                </button>
                <h2 class="hndle"><span>🔌 REST API</span></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="disable_rest_api">Disable REST API</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_rest_api" id="disable_rest_api" value="1" <?php checked($settings['disable_rest_api']); ?>>
                                        <strong>Disable REST API</strong><br>
                                        <span class="description">Completely blocks the REST API for non-logged users</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- 🔄 UPDATES -->
            <div class="postbox">
                <button type="button" class="handlediv" aria-expanded="true">
                    <span class="screen-reader-text">Toggle panel</span>
                    <span class="toggle-indicator" aria-hidden="true"></span>
                </button>
                <h2 class="hndle"><span>🔄 Updates & Maintenance</span></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="disable_auto_updates">Disable Auto-Updates</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_auto_updates" id="disable_auto_updates" value="1" <?php checked($settings['disable_auto_updates']); ?>>
                                        <strong>Disable auto-updates</strong><br>
                                        <span class="description">Blocks all automatic updates (core, plugins, themes, translations)</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="disable_update_checks">Disable Update Checks</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_update_checks" id="disable_update_checks" value="1" <?php checked($settings['disable_update_checks']); ?>>
                                        <strong>Disable update checks</strong><br>
                                        <span class="description">Blocks periodic checks for new updates</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="hide_update_notices">Hide Notices</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="hide_update_notices" id="hide_update_notices" value="1" <?php checked($settings['hide_update_notices']); ?>>
                                        <strong>Hide update notices</strong><br>
                                        <span class="description">Hides badges and update notifications in admin</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- 🗑️ ADDITIONAL LIBRARIES -->
            <div class="postbox">
                <button type="button" class="handlediv" aria-expanded="true">
                    <span class="screen-reader-text">Toggle panel</span>
                    <span class="toggle-indicator" aria-hidden="true"></span>
                </button>
                <h2 class="hndle"><span>🗑️ Additional Libraries</span></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="disable_ixr">Disable IXR</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_ixr" id="disable_ixr" value="1" <?php checked($settings['disable_ixr']); ?>>
                                        <strong>Disable IXR (XML-RPC Library)</strong><br>
                                        <span class="description">Removes the IXR library used by XML-RPC</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- 🚫 WORDPRESS BRANDING -->
            <div class="postbox">
                <button type="button" class="handlediv" aria-expanded="true">
                    <span class="screen-reader-text">Toggle panel</span>
                    <span class="toggle-indicator" aria-hidden="true"></span>
                </button>
                <h2 class="hndle"><span>🚫 WordPress Branding</span></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="remove_wp_branding">Remove Branding</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="remove_wp_branding" id="remove_wp_branding" value="1" <?php checked($settings['remove_wp_branding']); ?>>
                                        <strong>Remove WordPress branding</strong><br>
                                        <span class="description">Hides logo, favicon and all visual references to WordPress</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="remove_dashboard">Remove Dashboard</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="remove_dashboard" id="remove_dashboard" value="1" <?php checked($settings['remove_dashboard']); ?>>
                                        <strong>Remove dashboard page</strong><br>
                                        <span class="description">Removes the dashboard page from menu and redirects to "All Posts"</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- 🔗 UPLOAD REDIRECT -->
            <div class="postbox">
                <button type="button" class="handlediv" aria-expanded="true">
                    <span class="screen-reader-text">Toggle panel</span>
                    <span class="toggle-indicator" aria-hidden="true"></span>
                </button>
                <h2 class="hndle"><span>🔗 Upload Redirect</span></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="redirect_plugin_upload">Redirect Plugin Upload</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="redirect_plugin_upload" id="redirect_plugin_upload" value="1" <?php checked($settings['redirect_plugin_upload']); ?>>
                                        <strong>Redirect "Add Plugin"</strong><br>
                                        <span class="description">Redirects directly to Upload tab instead of repository</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="redirect_theme_upload">Redirect Theme Upload</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="redirect_theme_upload" id="redirect_theme_upload" value="1" <?php checked($settings['redirect_theme_upload']); ?>>
                                        <strong>Redirect "Add Theme"</strong><br>
                                        <span class="description">Redirects directly to Upload tab instead of repository</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- 🗑️ REVISIONS -->
            <div class="postbox">
                <button type="button" class="handlediv" aria-expanded="true">
                    <span class="screen-reader-text">Toggle panel</span>
                    <span class="toggle-indicator" aria-hidden="true"></span>
                </button>
                <h2 class="hndle"><span>🗑️ Revisions</span></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="disable_revisions">Disable Revisions</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_revisions" id="disable_revisions" value="1" <?php checked($settings['disable_revisions']); ?>>
                                        <strong>Disable revisions</strong><br>
                                        <span class="description">Blocks saving of post revisions (saves database space)</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- 🚫 ADMIN BAR -->
            <div class="postbox">
                <button type="button" class="handlediv" aria-expanded="true">
                    <span class="screen-reader-text">Toggle panel</span>
                    <span class="toggle-indicator" aria-hidden="true"></span>
                </button>
                <h2 class="hndle"><span>🚫 Admin Bar</span></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="disable_admin_bar">Disable Admin Bar</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_admin_bar" id="disable_admin_bar" value="1" <?php checked($settings['disable_admin_bar']); ?>>
                                        <strong>Disable admin bar</strong><br>
                                        <span class="description">Hides the admin bar in frontend</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- 📉 DASHBOARD WIDGETS -->
            <div class="postbox">
                <button type="button" class="handlediv" aria-expanded="true">
                    <span class="screen-reader-text">Toggle panel</span>
                    <span class="toggle-indicator" aria-hidden="true"></span>
                </button>
                <h2 class="hndle"><span>📉 Dashboard Widgets</span></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="remove_dashboard_widgets">Remove Widgets</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="remove_dashboard_widgets" id="remove_dashboard_widgets" value="1" <?php checked($settings['remove_dashboard_widgets']); ?>>
                                        <strong>Remove all widgets</strong><br>
                                        <span class="description">Removes all widgets from admin dashboard for a clean interface</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- ❓ HELP & SCREEN OPTIONS -->
            <div class="postbox">
                <button type="button" class="handlediv" aria-expanded="true">
                    <span class="screen-reader-text">Toggle panel</span>
                    <span class="toggle-indicator" aria-hidden="true"></span>
                </button>
                <h2 class="hndle"><span>❓ Help & Screen Options</span></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="disable_help_tabs">Disable Help Tabs</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_help_tabs" id="disable_help_tabs" value="1" <?php checked($settings['disable_help_tabs']); ?>>
                                        <strong>Disable help tabs</strong><br>
                                        <span class="description">Hides help tabs in top right corner on every admin page</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="disable_screen_options">Disable Screen Options</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_screen_options" id="disable_screen_options" value="1" <?php checked($settings['disable_screen_options']); ?>>
                                        <strong>Disable screen options</strong><br>
                                        <span class="description">Hides the "Screen Options" menu in top right corner</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- 🎯 POINTER JS -->
            <div class="postbox">
                <button type="button" class="handlediv" aria-expanded="true">
                    <span class="screen-reader-text">Toggle panel</span>
                    <span class="toggle-indicator" aria-hidden="true"></span>
                </button>
                <h2 class="hndle"><span>🎯 Pointer JS</span></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="disable_pointer_js">Disable Pointer JS</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_pointer_js" id="disable_pointer_js" value="1" <?php checked($settings['disable_pointer_js']); ?>>
                                        <strong>Disable pointer JS</strong><br>
                                        <span class="description">Removes WordPress interactive tooltips (e.g. "Welcome to dashboard")</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- 🎨 COLOR PICKER -->
            <div class="postbox">
                <button type="button" class="handlediv" aria-expanded="true">
                    <span class="screen-reader-text">Toggle panel</span>
                    <span class="toggle-indicator" aria-hidden="true"></span>
                </button>
                <h2 class="hndle"><span>🎨 Color Picker</span></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="disable_color_picker">Disable Color Picker</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_color_picker" id="disable_color_picker" value="1" <?php checked($settings['disable_color_picker']); ?>>
                                        <strong>Disable color picker</strong><br>
                                        <span class="description">Removes the JavaScript color picker from admin pages</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- 🔐 PASSWORD METER -->
            <div class="postbox">
                <button type="button" class="handlediv" aria-expanded="true">
                    <span class="screen-reader-text">Toggle panel</span>
                    <span class="toggle-indicator" aria-hidden="true"></span>
                </button>
                <h2 class="hndle"><span>🔐 Password Meter</span></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="disable_password_meter">Disable Password Meter</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_password_meter" id="disable_password_meter" value="1" <?php checked($settings['disable_password_meter']); ?>>
                                        <strong>Disable password strength meter</strong><br>
                                        <span class="description">Removes password strength check (useful for internal sites)</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- 📄 TEXT DIFF -->
            <div class="postbox">
                <button type="button" class="handlediv" aria-expanded="true">
                    <span class="screen-reader-text">Toggle panel</span>
                    <span class="toggle-indicator" aria-hidden="true"></span>
                </button>
                <h2 class="hndle"><span>📄 Text Diff</span></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="disable_text_diff">Disable Text Diff</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_text_diff" id="disable_text_diff" value="1" <?php checked($settings['disable_text_diff']); ?>>
                                        <strong>Disable text diff</strong><br>
                                        <span class="description">Removes the text comparison library used in revisions</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- 🔄 JSON2 -->
            <div class="postbox">
                <button type="button" class="handlediv" aria-expanded="true">
                    <span class="screen-reader-text">Toggle panel</span>
                    <span class="toggle-indicator" aria-hidden="true"></span>
                </button>
                <h2 class="hndle"><span>🔄 JSON2 (Legacy)</span></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="disable_json2">Disable JSON2</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="disable_json2" id="disable_json2" value="1" <?php checked($settings['disable_json2']); ?>>
                                        <strong>Disable JSON2</strong><br>
                                        <span class="description">Removes the legacy JSON polyfill for old browsers</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- 🧩 USELESS WIDGETS -->
            <div class="postbox">
                <button type="button" class="handlediv" aria-expanded="true">
                    <span class="screen-reader-text">Toggle panel</span>
                    <span class="toggle-indicator" aria-hidden="true"></span>
                </button>
                <h2 class="hndle"><span>🧩 Useless Widgets</span></h2>
                <div class="inside">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="remove_useless_widgets">Remove Useless Widgets</label></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="checkbox" name="remove_useless_widgets" id="remove_useless_widgets" value="1" <?php checked($settings['remove_useless_widgets']); ?>>
                                        <strong>Remove useless widgets</strong><br>
                                        <span class="description">Disables unnecessary widgets (Pages, Calendar, Archives, Links, Meta, Search, Text, Categories, Recent Posts, Tag Cloud, Menu, Custom HTML)</span>
                                    </label>
                                </fieldset>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- SAVE BUTTONS -->
            <div style="padding: 20px; background: #f9f9f9; border: 1px solid #e5e5e5; border-radius: 4px; margin-top: 20px;">
                <button type="submit" name="leanpress_save" class="button button-primary button-large">
                    💾 Save Settings
                </button>
                <a href="<?php echo wp_nonce_url(admin_url('admin.php?page=leanpress-control-panel&leanpress_reset=1'), 'leanpress_reset_settings'); ?>" 
                   class="button button-secondary button-large" 
                   onclick="return confirm('⚠️ Are you sure you want to restore all settings to default values? This action cannot be undone.');">
                    🔄 Restore Defaults
                </a>
            </div>
        </form>
    </div>
    
    <style>
        .leanpress-stats {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 20px;
            margin: 20px 0;
        }
        
        .leanpress-stat-box {
            padding: 20px;
            text-align: center;
        }
        
        .leanpress-stat-box > div {
            margin: 0;
        }
        
        .leanpress-stat-box > div:first-child {
            font-size: 36px;
            font-weight: bold;
        }
        
        .leanpress-stat-box > div:last-child {
            font-size: 14px;
            color: #646970;
            text-transform: uppercase;
            letter-spacing: 1px;
        }
        
        .postbox {
            background: #fff;
            border: 1px solid #ccd0d4;
            box-shadow: 0 1px 1px rgba(0,0,0,0.04);
            margin-bottom: 20px;
        }
        
        .postbox h2.hndle {
            font-size: 18px;
            padding: 12px 15px;
            margin: 0;
            line-height: 1.4;
            border-bottom: 1px solid #ccd0d4;
            cursor: move;
        }
        
        .postbox .inside {
            padding: 15px 20px;
        }
        
        .form-table th {
            padding: 15px 10px 15px 0;
            width: 250px;
            vertical-align: top;
            font-weight: 600;
        }
        
        .form-table td {
            padding: 15px 0;
            vertical-align: top;
        }
        
        .form-table label {
            display: block;
            margin-bottom: 8px;
            line-height: 1.5;
        }
        
        .form-table .description {
            color: #646970;
            font-size: 13px;
            display: block;
            margin-top: 4px;
            margin-left: 24px;
        }
        
        .form-table input[type="checkbox"] {
            margin-right: 8px;
            width: 16px;
            height: 16px;
            vertical-align: middle;
        }
        
        @media (max-width: 782px) {
            .form-table th {
                width: auto;
                display: block;
                padding-bottom: 10px;
            }
            
            .form-table td {
                display: block;
                padding-top: 0;
            }
        }
    </style>
    <?php
}

// ============================================================================
// 6. FEATURE IMPLEMENTATION
// ============================================================================

// 🔒 Security Headers
if (!is_admin() && !defined('DOING_AJAX') && !defined('WP_CLI') && !wp_doing_cron() && PHP_SAPI !== 'cli' && empty(headers_sent())) {
    $is_https = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ||
               (!empty($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] === 'https');
    header('X-Frame-Options: SAMEORIGIN');
    header('X-Content-Type-Options: nosniff');
    header('X-XSS-Protection: 1; mode=block');
    header('Referrer-Policy: strict-origin-when-cross-origin');
    if ($is_https) {
        header('Strict-Transport-Security: max-age=31536000; includeSubDomains; preload');
    }
}

// 🚀 JavaScript Optimizations

// Lazy Load jQuery
if (leanpress_is_enabled('lazy_load_jquery')) {
    add_action('wp_enqueue_scripts', function() {
        wp_deregister_script('jquery');
        wp_deregister_script('jquery-core');
    }, 1);
    
    add_action('wp_footer', function() {
        global $wp_scripts;
        
        $needs_jquery = false;
        foreach ($wp_scripts->queue as $handle) {
            $deps = $wp_scripts->registered[$handle]->deps ?? [];
            if (in_array('jquery', $deps) || in_array('jquery-core', $deps)) {
                $needs_jquery = true;
                break;
            }
        }
        
        if ($needs_jquery) {
            wp_enqueue_script('jquery');
        }
    }, 1);
}

// Remove jQuery Migrate
if (leanpress_is_enabled('remove_jquery_migrate')) {
    add_action('wp_enqueue_scripts', function() {
        wp_deregister_script('jquery-migrate');
    }, 9999);
}

// Remove wp-embed
if (leanpress_is_enabled('remove_wp_embed')) {
    add_action('wp_enqueue_scripts', function() {
        wp_deregister_script('wp-embed');
        wp_dequeue_script('wp-embed');
    }, 9999);
}

// Remove unnecessary scripts
add_action('wp_enqueue_scripts', function() {
    if (leanpress_is_enabled('disable_comments') || !comments_open()) {
        wp_deregister_script('comment-reply');
        wp_dequeue_script('comment-reply');
    }
}, 9999);

// Admin: Remove unnecessary scripts
add_action('admin_enqueue_scripts', function() {
    if (leanpress_is_enabled('disable_heartbeat')) {
        wp_deregister_script('heartbeat');
        wp_dequeue_script('heartbeat');
        wp_dequeue_script('autosave');
    }
    
    if (leanpress_is_enabled('disable_color_picker')) {
        wp_deregister_script('wp-color-picker');
        wp_dequeue_script('wp-color-picker');
        wp_dequeue_style('wp-color-picker');
    }
    
    if (leanpress_is_enabled('disable_pointer_js')) {
        wp_deregister_script('wp-pointer');
        wp_dequeue_script('wp-pointer');
        wp_dequeue_style('wp-pointer');
    }
    
    if (leanpress_is_enabled('disable_password_meter')) {
        wp_deregister_script('password-strength-meter');
        wp_dequeue_script('password-strength-meter');
    }
    
    if (leanpress_is_enabled('disable_text_diff')) {
        wp_deregister_script('textdiff');
        wp_dequeue_script('textdiff');
    }
    
    if (leanpress_is_enabled('disable_json2')) {
        wp_deregister_script('json2');
        wp_dequeue_script('json2');
    }
    
    // Remove media and customize scripts
    wp_dequeue_script('media-views');
    wp_dequeue_script('media-models');
    wp_dequeue_script('media-grid');
    wp_dequeue_script('customize-controls');
    wp_dequeue_script('customize-widgets');
    wp_dequeue_script('customize-nav-menus');
}, 9999);

// 🔌 Disable Gutenberg
if (leanpress_is_enabled('disable_gutenberg')) {
    add_filter('use_block_editor_for_post', '__return_false', 9999);
    add_filter('use_block_editor_for_post_type', '__return_false', 9999);
    
    add_action('admin_enqueue_scripts', function() {
        $gutenberg_assets = [
            'wp-block-library', 'wp-block-library-theme', 'wp-edit-post', 'wp-editor',
            'wp-format-library', 'wp-reusable-blocks', 'wp-block-directory',
            'wp-block-patterns', 'wp-block-serialization-default-parser',
            'wp-block-editor', 'wp-blocks', 'wp-edit-widgets', 'wp-customize-widgets'
        ];
        
        foreach ($gutenberg_assets as $handle) {
            wp_dequeue_script($handle);
            wp_dequeue_style($handle);
            wp_deregister_script($handle);
            wp_deregister_style($handle);
        }
    }, 9999);
    
    add_filter('rest_endpoints', function($endpoints) {
        $gutenberg_routes = [
            '/wp/v2/block-types', '/wp/v2/block-renderer', '/wp/v2/blocks',
            '/wp/v2/patterns', '/wp/v2/templates', '/wp/v2/template-parts',
            '/wp/v2/wp-blocks', '/wp/v2/widget-types', '/wp/v2/widgets'
        ];
        
        foreach ($gutenberg_routes as $route) {
            unset($endpoints[$route]);
        }
        return $endpoints;
    }, 9999);
    
    add_action('admin_menu', function() {
        remove_menu_page('gutenberg');
        remove_submenu_page('edit.php?post_type=wp_block', 'edit.php?post_type=wp_block');
        remove_submenu_page('edit.php?post_type=wp_template', 'edit.php?post_type=wp_template');
        remove_submenu_page('edit.php?post_type=wp_template_part', 'edit.php?post_type=wp_template_part');
    }, 9999);
    
    remove_filter('the_content', 'do_blocks', 9);
    
    add_action('wp_enqueue_scripts', function() {
        wp_dequeue_script('wp-block-serialization-default-parser');
    }, 9999);
    
    add_action('after_setup_theme', function() {
        remove_theme_support('block-templates');
        remove_theme_support('block-template-parts');
        remove_theme_support('core-block-patterns');
        remove_theme_support('widgets-block-editor');
    }, 0);
    
    add_filter('gutenberg_use_widgets_block_editor', '__return_false', 9999);
    add_filter('use_widgets_block_editor', '__return_false', 9999);
}

// 🔌 Disable TinyMCE + Pure HTML Textarea
if (leanpress_is_enabled('disable_tinymce') || leanpress_is_enabled('use_html_editor')) {
    add_filter('tiny_mce_before_init', '__return_false', 9999);
    
    add_filter('wp_editor_settings', function($settings) {
        $settings['tinymce'] = false;
        $settings['quicktags'] = false;
        $settings['media_buttons'] = false;
        return $settings;
    }, 9999);
    
    add_action('admin_enqueue_scripts', function() {
        wp_deregister_script('wp-tinymce');
        wp_deregister_script('tinymce');
        wp_deregister_script('wp-tinymce-root');
        wp_dequeue_script('wp-tinymce');
        wp_dequeue_script('tinymce');
        wp_dequeue_script('wp-tinymce-root');
        wp_dequeue_style('wp-editor');
        wp_dequeue_style('wp-auth-check');
        wp_dequeue_script('quicktags');
        wp_dequeue_script('editor');
    }, 9999);
    
    add_action('init', function() {
        remove_action('admin_print_footer_scripts', 'wp_tiny_mce', 50);
        remove_action('admin_head', 'wp_print_styles', 20);
        remove_action('admin_head', 'wp_print_editor_js', 50);
        remove_action('admin_head', 'wp_admin_canonical_url');
        remove_action('admin_print_scripts', 'wp_print_scripts');
        remove_action('admin_print_styles', 'wp_print_styles');
    }, 9999);
    
    if (leanpress_is_enabled('use_html_editor')) {
        add_action('edit_form_after_title', function($post) {
            if (in_array(get_post_type($post), ['post', 'page'])) {
                remove_post_type_support(get_post_type($post), 'editor');
                echo '<div id="lb-editor-container" style="margin: 20px 0; padding: 0;">';
                echo '<label for="lb-editor" style="display: block; margin-bottom: 8px; font-weight: 600;">Content (HTML)</label>';
                echo '<textarea name="content" id="lb-editor" style="width: 100%; min-height: 600px; font-family: \'Courier New\', monospace; font-size: 14px; line-height: 1.5; padding: 15px; border: 2px solid #e1e1e1; border-radius: 4px; resize: vertical; box-sizing: border-box;" autocomplete="off">' . esc_textarea($post->post_content) . '</textarea>';
                echo '</div>';
            }
        }, 10, 1);
        
        add_action('save_post', function($post_id, $post, $update) {
            if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
            if (!current_user_can('edit_post', $post_id)) return;
            
            if (isset($_POST['content'])) {
                $content = wp_kses_post(wp_unslash($_POST['content']));
                remove_action('save_post', __FUNCTION__, 10);
                wp_update_post(['ID' => $post_id, 'post_content' => $content]);
                add_action('save_post', __FUNCTION__, 10, 3);
            }
        }, 10, 3);
        
        add_action('admin_menu', function() {
            remove_meta_box('postdivrich', 'post', 'normal');
            remove_meta_box('postdivrich', 'page', 'normal');
        }, 9999);
        
        add_action('admin_head', function() {
            echo '<style>
                #postdivrich, #postdiv { display: none !important; }
                #lb-editor { transition: border-color 0.3s ease; }
                #lb-editor:focus { border-color: #667eea !important; box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1); outline: none; }
                .insert-media { display: none !important; }
                #content-tmce, #content-html { display: none !important; }
            </style>';
        });
    }
}

// 🗑️ Remove jQuery UI
if (leanpress_is_enabled('disable_jquery_ui')) {
    add_action('wp_enqueue_scripts', 'leanpress_remove_jquery_ui_frontend', 9999);
    add_action('admin_enqueue_scripts', 'leanpress_remove_jquery_ui_admin', 9999);
    
    function leanpress_remove_jquery_ui_frontend() {
        $jquery_ui_components = [
            'jquery-ui-core', 'jquery-ui-widget', 'jquery-ui-mouse', 'jquery-ui-position',
            'jquery-ui-draggable', 'jquery-ui-droppable', 'jquery-ui-resizable',
            'jquery-ui-selectable', 'jquery-ui-sortable', 'jquery-ui-accordion',
            'jquery-ui-autocomplete', 'jquery-ui-button', 'jquery-ui-datepicker',
            'jquery-ui-dialog', 'jquery-ui-menu', 'jquery-ui-progressbar',
            'jquery-ui-selectmenu', 'jquery-ui-slider', 'jquery-ui-spinner',
            'jquery-ui-tabs', 'jquery-ui-tooltip', 'jquery-ui-effect',
            'jquery-effects-core', 'jquery-effects-blind', 'jquery-effects-bounce',
            'jquery-effects-clip', 'jquery-effects-drop', 'jquery-effects-explode',
            'jquery-effects-fade', 'jquery-effects-fold', 'jquery-effects-highlight',
            'jquery-effects-puff', 'jquery-effects-pulsate', 'jquery-effects-scale',
            'jquery-effects-shake', 'jquery-effects-size', 'jquery-effects-slide',
            'jquery-effects-transfer'
        ];
        
        foreach ($jquery_ui_components as $handle) {
            wp_deregister_script($handle);
            wp_dequeue_script($handle);
        }
    }
    
    function leanpress_remove_jquery_ui_admin() {
        $jquery_ui_components = [
            'jquery-ui-core', 'jquery-ui-widget', 'jquery-ui-mouse', 'jquery-ui-position',
            'jquery-ui-draggable', 'jquery-ui-droppable', 'jquery-ui-resizable',
            'jquery-ui-selectable', 'jquery-ui-sortable', 'jquery-ui-accordion',
            'jquery-ui-autocomplete', 'jquery-ui-button', 'jquery-ui-datepicker',
            'jquery-ui-dialog', 'jquery-ui-menu', 'jquery-ui-progressbar',
            'jquery-ui-selectmenu', 'jquery-ui-slider', 'jquery-ui-spinner',
            'jquery-ui-tabs', 'jquery-ui-tooltip', 'jquery-ui-effect',
            'jquery-effects-core', 'jquery-effects-blind', 'jquery-effects-bounce',
            'jquery-effects-clip', 'jquery-effects-drop', 'jquery-effects-explode',
            'jquery-effects-fade', 'jquery-effects-fold', 'jquery-effects-highlight',
            'jquery-effects-puff', 'jquery-effects-pulsate', 'jquery-effects-scale',
            'jquery-effects-shake', 'jquery-effects-size', 'jquery-effects-slide',
            'jquery-effects-transfer'
        ];
        
        foreach ($jquery_ui_components as $handle) {
            wp_deregister_script($handle);
            wp_dequeue_script($handle);
        }
    }
}

// 💬 Remove Comments
if (leanpress_is_enabled('disable_comments')) {
    add_filter('comments_open', '__return_false', 9999, 2);
    add_filter('pings_open', '__return_false', 9999, 2);
    add_filter('comments_array', '__return_empty_array', 9999, 2);
    add_filter('pre_comment_approved', '__return_false', 9999);
    add_filter('get_comments_number', '__return_zero');
    add_filter('comments_number', '__return_empty_string');
    add_filter('comments_template', '__return_empty_string', 9999);
    add_filter('pre_option_default_comment_status', '__return_false');
    add_filter('pre_option_default_ping_status', '__return_false');
    
    add_action('admin_init', function() {
        foreach (get_post_types() as $post_type) {
            remove_post_type_support($post_type, 'comments');
            remove_post_type_support($post_type, 'trackbacks');
        }
        
        if (!get_option('leanpress_comments_closed_once')) {
            global $wpdb;
            $wpdb->query("
                UPDATE {$wpdb->posts}
                SET comment_status = 'closed', ping_status = 'closed'
                WHERE comment_status = 'open' OR ping_status = 'open'
            ");
            update_option('leanpress_comments_closed_once', 1);
        }
        
        wp_clear_scheduled_hook('wp_scheduled_delete');
    }, 9999);
    
    add_action('widgets_init', function() {
        unregister_widget('WP_Widget_Recent_Comments');
        unregister_widget('WP_Widget_Comments');
    }, 9999);
    
    add_filter('rest_endpoints', function($endpoints) {
        unset($endpoints['/wp/v2/comments']);
        unset($endpoints['/wp/v2/comments/(?P<id>[\d]+)']);
        return $endpoints;
    }, 9999);
    
    add_action('admin_menu', function() {
        remove_menu_page('edit-comments.php');
        remove_meta_box('commentstatusdiv', 'post', 'normal');
        remove_meta_box('commentsdiv', 'post', 'normal');
        remove_meta_box('commentstatusdiv', 'page', 'normal');
        remove_meta_box('commentsdiv', 'page', 'normal');
        
        foreach (get_post_types() as $post_type) {
            remove_meta_box('commentstatusdiv', $post_type, 'normal');
            remove_meta_box('commentsdiv', $post_type, 'normal');
        }
        
        remove_submenu_page('options-general.php', 'options-discussion.php');
    }, 9999);
    
    add_action('admin_head', function() {
        echo '<style>
            .column-comments, .post-com-count, .post-com-count-wrapper,
            #adminmenu .awaiting-mod, #adminmenu .comment-count,
            .manage-column.column-comments { display: none !important; }
        </style>';
    });
}

// 🗑️ Remove RSS Feeds
if (leanpress_is_enabled('disable_rss')) {
    add_action('parse_request', function($wp) {
        $uri = $_SERVER['REQUEST_URI'] ?? '';
        if (preg_match('!/feed(/.*)?$!i', $uri)) {
            status_header(404);
            die();
        }
    }, 1);
    
    add_action('init', function() {
        remove_action('wp_head', 'feed_links', 2);
        remove_action('wp_head', 'feed_links_extra', 3);
    }, 9999);
    
    add_action('init', function() {
        add_action('do_feed', function() {
            wp_die('RSS Feed disabled.');
        }, 1);
        add_action('do_feed_rdf', function() {
            wp_die('RSS Feed disabled.');
        }, 1);
        add_action('do_feed_rss', function() {
            wp_die('RSS Feed disabled.');
        }, 1);
        add_action('do_feed_rss2', function() {
            wp_die('RSS Feed disabled.');
        }, 1);
        add_action('do_feed_atom', function() {
            wp_die('RSS Feed disabled.');
        }, 1);
    }, 9999);
}

// 🗑️ Remove SimplePie
if (leanpress_is_enabled('disable_simplepie')) {
    add_action('wp_enqueue_scripts', 'leanpress_remove_simplepie_frontend', 9999);
    add_action('admin_enqueue_scripts', 'leanpress_remove_simplepie_admin', 9999);
    
    function leanpress_remove_simplepie_frontend() {
        wp_deregister_script('simplepie');
        wp_dequeue_script('simplepie');
    }
    
    function leanpress_remove_simplepie_admin() {
        wp_deregister_script('simplepie');
        wp_dequeue_script('simplepie');
    }
    
    add_filter('wp_feed_cache_transient_lifetime', '__return_zero');
    add_filter('wp_feed_fetch_feed_block', '__return_false');
    add_filter('fetch_feed', '__return_false', 9999);
    
    add_action('widgets_init', function() {
        unregister_widget('WP_Widget_RSS');
    }, 9999);
}

// ❤️ Remove Heartbeat API
if (leanpress_is_enabled('disable_heartbeat')) {
    add_action('init', function() {
        wp_deregister_script('heartbeat');
        add_filter('heartbeat_settings', '__return_false');
        remove_action('admin_enqueue_scripts', 'wp_heartbeat_set_suspension');
        remove_action('wp_ajax_heartbeat', 'wp_ajax_heartbeat');
        remove_action('wp_ajax_nopriv_heartbeat', 'wp_ajax_heartbeat');
    }, 9999);
}

// 🚫 Disable WordPress.org Repository
if (leanpress_is_enabled('disable_wporg_repository')) {
    add_filter('plugins_api', function($result, $action, $args) {
        if (in_array($action, ['query_plugins', 'plugin_information'])) {
            return new WP_Error('disabled', 'Repository disabled. Use manual upload.');
        }
        return $result;
    }, 10, 3);
    
    add_filter('themes_api', function($result, $action, $args) {
        if (in_array($action, ['query_themes', 'theme_information'])) {
            return new WP_Error('disabled', 'Repository disabled. Use manual upload.');
        }
        return $result;
    }, 10, 3);
    
    add_filter('pre_set_site_transient_update_plugins', function($transient) {
        if (!is_object($transient)) $transient = new stdClass;
        if (isset($transient->response)) $transient->response = [];
        return $transient;
    }, 9999);
    
    add_filter('pre_set_site_transient_update_themes', function($transient) {
        if (!is_object($transient)) $transient = new stdClass;
        if (isset($transient->response)) $transient->response = [];
        return $transient;
    }, 9999);
    
    add_action('admin_head', function() {
        global $pagenow;
        if ($pagenow === 'plugin-install.php') {
            echo '<style>
                .upload-plugin-tab { display: block !important; }
                .plugin-install-php .wp-filter,
                .plugin-install-php .plugin-card { display: none !important; }
                .plugin-install-php .upload-plugin { display: block !important; }
            </style>';
        }
    }, 9999);
}

// 🔒 Block External Connections
if (leanpress_is_enabled('block_external_connections')) {
    add_filter('pre_http_request', function($preempt, $r, $url) {
        $blocked_domains = [
            'wordpress.org', 'api.wordpress.org', 'wordpress.com', 'wp.com',
            'gravatar.com', 's.w.org', 'fonts.googleapis.com', 'ajax.googleapis.com',
            'code.jquery.com', 'cdnjs.cloudflare.com', 'jsdelivr.net', 'unpkg.com'
        ];
        
        foreach ($blocked_domains as $domain) {
            if (strpos($url, $domain) !== false) {
                return new WP_Error('external_blocked', 'External connection blocked.');
            }
        }
        
        return $preempt;
    }, 10, 3);
}

// 🔒 Disable XML-RPC
if (leanpress_is_enabled('disable_xmlrpc')) {
    add_filter('xmlrpc_enabled', '__return_false');
    add_filter('pre_option_enable_xmlrpc', '__return_false');
    
    add_action('init', function() {
        if (defined('XMLRPC_REQUEST') && XMLRPC_REQUEST) {
            status_header(403);
            die('XML-RPC disabled.');
        }
        
        add_action('xmlrpc_call', function($method) {
            if ($method === 'pingback.ping') {
                wp_die('Pingback disabled.', 'Access denied', ['response' => 403]);
            }
        }, 1);
        
        add_filter('enable_enclosures', '__return_false');
        add_filter('pre_update_option_ping_sites', '__return_empty_array');
    }, 1);
    
    add_action('init', function() {
        add_filter('pings_open', '__return_false', 9999, 2);
        add_filter('pre_option_default_pingback_flag', '__return_zero');
        add_filter('pre_option_default_ping_status', '__return_zero');
    }, 9999);
    
    add_action('wp_enqueue_scripts', function() {
        wp_deregister_script('xmlrpc');
        wp_dequeue_script('xmlrpc');
    }, 9999);
    
    add_action('admin_enqueue_scripts', function() {
        wp_deregister_script('xmlrpc');
        wp_dequeue_script('xmlrpc');
    }, 9999);
}

// 🔒 Remove WP Generator
if (leanpress_is_enabled('remove_wp_generator')) {
    add_action('init', function() {
        remove_action('wp_head', 'wp_generator');
        remove_action('wp_head', 'rsd_link');
        remove_action('wp_head', 'wlwmanifest_link');
        remove_action('wp_head', 'index_rel_link');
        remove_action('wp_head', 'parent_post_rel_link');
        remove_action('wp_head', 'start_post_rel_link');
        remove_action('wp_head', 'adjacent_posts_rel_link_wp_head');
        remove_action('wp_head', 'rest_output_link_wp_head');
        remove_action('wp_head', 'wp_oembed_add_discovery_links');
        remove_action('wp_head', 'wp_oembed_add_host_js');
        remove_action('wp_head', 'wp_shortlink_wp_head');
        remove_action('template_redirect', 'wp_shortlink_header');
        remove_action('template_redirect', 'rest_output_link_header');
        
        add_filter('style_loader_src', function($src) {
            return remove_query_arg('ver', $src);
        });
        add_filter('script_loader_src', function($src) {
            return remove_query_arg('ver', $src);
        });
        
        header_remove('X-Pingback');
    }, 9999);
}

// 🔒 Disable oEmbed
if (leanpress_is_enabled('disable_oembed')) {
    add_action('init', function() {
        remove_action('wp_head', 'wp_oembed_add_discovery_links');
        remove_action('wp_head', 'wp_oembed_add_host_js');
        remove_filter('oembed_dataparse', 'wp_filter_oembed_result');
        remove_filter('pre_oembed_result', 'wp_filter_pre_oembed_result');
        remove_filter('rest_endpoints', 'WP_oEmbed_Controller::register_routes');
        
        add_filter('embed_oembed_discover', '__return_false');
        add_filter('oembed_dataparse', '__return_false');
        add_filter('embed_maybe_embed', '__return_false');
    }, 9999);
}

// 🔒 Disable WP-Cron
if (leanpress_is_enabled('disable_wp_cron')) {
    add_filter('disable_wp_cron', '__return_true');
    add_filter('pre_option_enable_wp_cron', '__return_false');
    
    add_action('init', function() {
        remove_action('init', 'wp_cron');
        remove_action('wp_head', 'wp_cron');
        
        if (strpos($_SERVER['REQUEST_URI'] ?? '', 'wp-cron.php') !== false) {
            status_header(403);
            die('WP-Cron disabled.');
        }
    }, 1);
}

// 🔒 Disable Application Passwords
if (leanpress_is_enabled('disable_app_passwords')) {
    add_filter('wp_is_application_passwords_available', '__return_false');
    add_filter('wp_is_application_passwords_available_for_user', '__return_false');
}

// 🔒 Disable Recovery Mode
if (leanpress_is_enabled('disable_recovery_mode')) {
    add_filter('wp_recovery_mode_active', '__return_false');
    add_filter('wp_force_recovery_mode', '__return_false');
}

// 🔒 Disable Site Health
if (leanpress_is_enabled('disable_site_health')) {
    add_action('admin_menu', function() {
        remove_submenu_page('tools.php', 'site-health.php');
    }, 9999);
    
    add_action('wp_dashboard_setup', function() {
        remove_meta_box('dashboard_site_health', 'dashboard', 'normal');
    }, 9999);
}

// 🔌 Disable REST API
if (leanpress_is_enabled('disable_rest_api')) {
    add_filter('rest_authentication_errors', function($result) {
        if (!empty($result)) return $result;
        if (!is_user_logged_in()) {
            return new WP_Error('rest_forbidden', 'REST API access denied.', ['status' => 403]);
        }
        return $result;
    }, 9999);
    
    add_filter('rest_endpoints', function($endpoints) {
        return [];
    }, 9999);
    
    add_action('parse_request', function($wp) {
        $uri = $_SERVER['REQUEST_URI'] ?? '';
        if (strpos($uri, '/wp-json/') === 0 || strpos($uri, '?rest_route=') !== false) {
            status_header(403);
            die('REST API disabled.');
        }
    }, 1);
    
    add_action('wp_enqueue_scripts', function() {
        wp_dequeue_script('wp-api');
        wp_dequeue_script('wp-api-request');
    }, 9999);
}

// 🔄 Disable Auto-Updates
if (leanpress_is_enabled('disable_auto_updates')) {
    add_filter('automatic_updater_disabled', '__return_true', 9999);
    add_filter('auto_update_plugin', '__return_false', 9999);
    add_filter('auto_update_theme', '__return_false', 9999);
    add_filter('auto_update_core', '__return_false', 9999);
    add_filter('auto_update_translation', '__return_false', 9999);
}

// 🔄 Disable Update Checks
if (leanpress_is_enabled('disable_update_checks')) {
    add_filter('pre_transient_update_core', '__return_null', 9999);
    add_filter('pre_site_transient_update_core', '__return_null', 9999);
    add_filter('pre_transient_update_plugins', '__return_null', 9999);
    add_filter('pre_site_transient_update_plugins', '__return_null', 9999);
    add_filter('pre_transient_update_themes', '__return_null', 9999);
    add_filter('pre_site_transient_update_themes', '__return_null', 9999);
    
    add_action('init', function() {
        remove_action('wp_version_check', 'wp_version_check');
        remove_action('wp_update_plugins', 'wp_update_plugins');
        remove_action('wp_update_themes', 'wp_update_themes');
        remove_action('wp_maybe_auto_update', 'wp_maybe_auto_update');
    }, 9999);
}

// 🔄 Hide Update Notices
if (leanpress_is_enabled('hide_update_notices')) {
    add_action('admin_menu', function() {
        remove_submenu_page('index.php', 'update-core.php');
    }, 9999);
    
    add_action('admin_init', function() {
        remove_action('admin_notices', 'update_nag', 3);
        remove_action('network_admin_notices', 'update_nag', 3);
        remove_action('admin_notices', 'maintenance_nag', 10);
        remove_action('network_admin_notices', 'maintenance_nag', 10);
    }, 9999);
    
    add_action('admin_head', function() {
        echo '<style>
            #menu-dashboard .update-plugins, #menu-plugins .update-plugins,
            #menu-appearance .update-plugins, .plugin-count,
            #wp-admin-bar-updates { display: none !important; }
        </style>';
    });
    
    add_filter('update_footer', '__return_empty_string', 9999);
    add_filter('admin_footer_text', '__return_empty_string', 9999);
    
    add_filter('auto_core_update_send_email', '__return_false', 9999);
    add_filter('auto_plugin_update_send_email', '__return_false', 9999);
    add_filter('auto_theme_update_send_email', '__return_false', 9999);
}

// 🗑️ Remove IXR
if (leanpress_is_enabled('disable_ixr')) {
    // Already handled in disable_xmlrpc
}

// 🚫 Remove WordPress Branding
if (leanpress_is_enabled('remove_wp_branding')) {
    add_action('init', function() {
        // Emoji
        remove_action('wp_head', 'print_emoji_detection_script', 7);
        remove_action('wp_print_styles', 'print_emoji_styles');
        remove_filter('the_content_feed', 'wp_staticize_emoji');
        remove_filter('comment_text_rss', 'wp_staticize_emoji');
    }, 1);
    
    add_action('admin_head', function() {
        echo '<style>
            /* Hide WP logo everywhere */
            img[src*="wordpress-logo"],
            img[src*="wp-logo"],
            .wp-badge,
            #wpadminbar .ab-icon,
            .about-text { display: none !important; }
            /* Remove icon from menu */
            .menu-icon-dashboard .wp-menu-image { background: none !important; }
            .menu-icon-dashboard .wp-menu-image::before { content: "" !important; }
        </style>';
    });
    
    add_filter('admin_title', fn($title) => str_replace(' &#8212; WordPress', '', $title));
    
    add_action('wp_head', function() {
        echo '<link rel="icon" href="image/svg+xml,<svg xmlns=%22http://www.w3.org/2000/svg%22 viewBox=%220 0 100 100%22><rect width=%22100%22 height=%22100%22 fill=%22%23fff%22/><text x=%2250%22 y=%2265%22 text-anchor=%22middle%22 font-size=%2240%22 fill=%22%23333%22>?</text></svg>" type="image/svg+xml">';
    }, 1);
    
    add_filter('the_content', function($content) {
        return preg_replace(
            '~<a[^>]*href=["\']https?://(?:www\.)?wordpress\.org[^"\']*["\'][^>]*>(.*?)</a>~i',
            '$1',
            $content
        );
    }, 9999);
}

// 🚫 Remove Dashboard
if (leanpress_is_enabled('remove_dashboard')) {
    add_action('admin_menu', function() {
        remove_menu_page('index.php');
    }, 9999);
    
    add_action('admin_init', function() {
        global $pagenow;
        if ($pagenow === 'index.php') {
            wp_redirect(admin_url('edit.php'));
            exit;
        }
    }, 1);
}

// 🔗 Redirect Plugin Upload
if (leanpress_is_enabled('redirect_plugin_upload')) {
    add_filter('admin_url', function($url, $path, $blog_id) {
        if ($path === 'plugin-install.php') {
            return admin_url('plugin-install.php?tab=upload');
        }
        return $url;
    }, 10, 3);
}

// 🔗 Redirect Theme Upload
if (leanpress_is_enabled('redirect_theme_upload')) {
    add_filter('admin_url', function($url, $path, $blog_id) {
        if ($path === 'theme-install.php') {
            return admin_url('theme-install.php?tab=upload');
        }
        return $url;
    }, 10, 3);
}

// 🗑️ Disable Revisions
if (leanpress_is_enabled('disable_revisions')) {
    define('WP_POST_REVISIONS', false);
    add_filter('wp_revisions_to_keep', '__return_zero', 9999);
}

// 🚫 Disable Admin Bar
if (leanpress_is_enabled('disable_admin_bar')) {
    add_action('after_setup_theme', function() {
        show_admin_bar(false);
    }, 9999);
    
    add_action('init', function() {
        remove_action('wp_footer', 'wp_admin_bar_render', 1000);
        remove_action('admin_footer', 'wp_admin_bar_render', 1000);
    }, 9999);
    
    add_action('wp_enqueue_scripts', function() {
        wp_dequeue_style('admin-bar');
        wp_dequeue_script('admin-bar');
    }, 9999);
}

// 📉 Remove Dashboard Widgets
if (leanpress_is_enabled('remove_dashboard_widgets')) {
    add_action('wp_dashboard_setup', function() {
        global $wp_meta_boxes;
        $wp_meta_boxes['dashboard'] = [];
    }, 9999);
}

// ❓ Disable Help Tabs
if (leanpress_is_enabled('disable_help_tabs')) {
    add_action('admin_head', function() {
        echo '<style>#contextual-help-link-wrap { display: none !important; }</style>';
    });
    
    add_filter('contextual_help', '__return_empty_string', 9999, 3);
}

// ❓ Disable Screen Options
if (leanpress_is_enabled('disable_screen_options')) {
    add_action('admin_head', function() {
        echo '<style>#screen-options-link-wrap { display: none !important; }</style>';
    });
    
    add_filter('screen_options_show_screen', '__return_false', 9999);
}

// 🎯 Disable Pointer JS
if (leanpress_is_enabled('disable_pointer_js')) {
    add_action('admin_enqueue_scripts', function() {
        wp_deregister_script('wp-pointer');
        wp_dequeue_script('wp-pointer');
        wp_dequeue_style('wp-pointer');
    }, 9999);
}

// 🎨 Disable Color Picker
if (leanpress_is_enabled('disable_color_picker')) {
    add_action('admin_enqueue_scripts', function() {
        wp_deregister_script('wp-color-picker');
        wp_dequeue_script('wp-color-picker');
        wp_dequeue_style('wp-color-picker');
    }, 9999);
}

// 🔐 Disable Password Meter
if (leanpress_is_enabled('disable_password_meter')) {
    add_action('admin_enqueue_scripts', function() {
        wp_deregister_script('password-strength-meter');
        wp_dequeue_script('password-strength-meter');
    }, 9999);
    
    add_action('wp_enqueue_scripts', function() {
        wp_dequeue_script('password-strength-meter');
    }, 9999);
}

// 📄 Disable Text Diff
if (leanpress_is_enabled('disable_text_diff')) {
    add_action('admin_enqueue_scripts', function() {
        wp_deregister_script('textdiff');
        wp_dequeue_script('textdiff');
    }, 9999);
}

// 🔄 Disable JSON2
if (leanpress_is_enabled('disable_json2')) {
    add_action('wp_enqueue_scripts', function() {
        wp_deregister_script('json2');
        wp_dequeue_script('json2');
    }, 9999);
    
    add_action('admin_enqueue_scripts', function() {
        wp_dequeue_script('json2');
    }, 9999);
}

// 🧩 Remove Useless Widgets
if (leanpress_is_enabled('remove_useless_widgets')) {
    add_action('widgets_init', function() {
        $useless_widgets = [
            'WP_Widget_Pages',
            'WP_Widget_Calendar',
            'WP_Widget_Archives',
            'WP_Widget_Links',
            'WP_Widget_Meta',
            'WP_Widget_Search',
            'WP_Widget_Text',
            'WP_Widget_Categories',
            'WP_Widget_Recent_Posts',
            'WP_Widget_Tag_Cloud',
            'WP_Nav_Menu_Widget',
            'WP_Widget_Custom_HTML'
        ];
        
        foreach ($useless_widgets as $widget) {
            unregister_widget($widget);
        }
    }, 9999);
}
?>